import { useState, useEffect } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { toast } from 'sonner';
import { Cake, ArrowLeft, Search, CheckCircle2, Circle, Clock } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const TrackOrderPage = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [trackingCode, setTrackingCode] = useState(searchParams.get('code') || '');
  const [order, setOrder] = useState(null);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    const code = searchParams.get('code');
    const id = searchParams.get('id');
    if (code) {
      setTrackingCode(code);
      handleTrackByCode(code);
    } else if (id) {
      handleTrackById(id);
    }
  }, []);

  const handleTrackByCode = async (code = trackingCode) => {
    if (!code) {
      toast.error('Please enter a tracking code');
      return;
    }

    setLoading(true);
    try {
      const response = await axios.get(`${API}/orders/tracking/${code.toUpperCase()}`);
      setOrder(response.data);
      toast.success('Order found!');
    } catch (error) {
      console.error('Error tracking order:', error);
      toast.error(error.response?.status === 404 ? 'Order not found with this tracking code' : 'Failed to track order');
      setOrder(null);
    } finally {
      setLoading(false);
    }
  };

  const handleTrackById = async (id) => {
    setLoading(true);
    try {
      const response = await axios.get(`${API}/orders/${id}`);
      setOrder(response.data);
      setTrackingCode(response.data.tracking_code);
    } catch (error) {
      console.error('Error tracking order:', error);
      toast.error('Failed to track order');
      setOrder(null);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen py-8 px-4 sm:px-6 lg:px-8 bg-white">
      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <Button
          variant="ghost"
          onClick={() => navigate('/')}
          className="mb-8 text-purple-600 hover:text-purple-700 hover:bg-purple-50"
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Back to Home
        </Button>

        <div className="text-center mb-12">
          <Cake className="w-16 h-16 text-purple-600 mx-auto mb-4" />
          <h1 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-2">Track Your Order</h1>
          <p className="text-gray-600">Enter your tracking code to see your order status</p>
        </div>

        {/* Search Form */}
        <div className="bg-white border border-gray-100 rounded-3xl p-8 shadow-lg mb-8">
          <div className="flex gap-4">
            <div className="flex-1">
              <Label htmlFor="tracking_code" className="text-gray-700 mb-2 block">Tracking Code</Label>
              <Input
                id="tracking_code"
                value={trackingCode}
                onChange={(e) => setTrackingCode(e.target.value.toUpperCase())}
                placeholder="CC-XXXX-XXXX"
                className="text-lg"
                onKeyPress={(e) => e.key === 'Enter' && handleTrackByCode()}
              />
            </div>
            <div className="flex items-end">
              <Button
                onClick={() => handleTrackByCode()}
                disabled={loading}
                className="bg-purple-600 hover:bg-purple-700 text-white px-8 py-6 rounded-full"
              >
                <Search className="w-5 h-5 mr-2" />
                {loading ? 'Searching...' : 'Track'}
              </Button>
            </div>
          </div>
        </div>

        {/* Order Details */}
        {order && (
          <div className="space-y-8">
            {/* Order Info Card */}
            <div className="bg-gradient-to-br from-purple-50 to-white border border-purple-100 rounded-3xl p-8 shadow-lg">
              <div className="flex justify-between items-start mb-6">
                <div>
                  <h2 className="text-2xl font-bold text-gray-900 mb-2">Order Details</h2>
                  <p className="text-purple-600 font-mono text-lg font-bold">
                    Tracking Code: {order.tracking_code}
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-sm text-gray-600">Order Date</p>
                  <p className="font-semibold">{new Date(order.created_at).toLocaleDateString()}</p>
                </div>
              </div>

              <div className="grid sm:grid-cols-2 gap-6">
                <div>
                  <p className="text-sm text-gray-600">Customer Name</p>
                  <p className="font-semibold text-gray-900">{order.customer_name}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Phone</p>
                  <p className="font-semibold text-gray-900">{order.phone}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Delivery Address</p>
                  <p className="font-semibold text-gray-900">{order.delivery_address}</p>
                </div>
                {order.delivery_date && (
                  <div>
                    <p className="text-sm text-gray-600">Delivery Date</p>
                    <p className="font-semibold text-gray-900">{new Date(order.delivery_date).toLocaleDateString()}</p>
                  </div>
                )}
              </div>
            </div>

            {/* Tracking Timeline */}
            <div className="bg-white border border-gray-100 rounded-3xl p-8 shadow-lg">
              <h2 className="text-2xl font-bold text-gray-900 mb-8">Order Progress</h2>
              
              <div className="relative">
                {/* Progress Line */}
                <div className="absolute left-6 top-0 bottom-0 w-0.5 bg-gray-200"></div>
                
                {/* Stages */}
                <div className="space-y-6">
                  {order.tracking_stages && order.tracking_stages.map((stage, index) => (
                    <div key={index} className="relative flex items-start gap-4">
                      {/* Icon */}
                      <div className={`relative z-10 flex items-center justify-center w-12 h-12 rounded-full ${
                        stage.completed 
                          ? 'bg-purple-600 text-white' 
                          : 'bg-white border-2 border-gray-300 text-gray-400'
                      }`}>
                        {stage.completed ? (
                          <CheckCircle2 className="w-6 h-6" />
                        ) : (
                          <Circle className="w-6 h-6" />
                        )}
                      </div>

                      {/* Content */}
                      <div className="flex-1 pb-8">
                        <div className={`p-4 rounded-2xl ${
                          stage.completed 
                            ? 'bg-purple-50 border border-purple-200' 
                            : 'bg-gray-50 border border-gray-200'
                        }`}>
                          <h3 className={`font-semibold text-lg ${
                            stage.completed ? 'text-purple-900' : 'text-gray-600'
                          }`}>
                            {stage.stage}
                          </h3>
                          {stage.timestamp && (
                            <p className="text-sm text-gray-600 mt-1 flex items-center gap-2">
                              <Clock className="w-4 h-4" />
                              {new Date(stage.timestamp).toLocaleString()}
                            </p>
                          )}
                          {!stage.completed && index > 0 && (
                            <p className="text-sm text-gray-500 mt-2 italic">
                              Pending...
                            </p>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </div>

            {/* Cake Details Card */}
            <div className="bg-white border border-gray-100 rounded-3xl p-8 shadow-lg">
              <h2 className="text-2xl font-bold text-gray-900 mb-6">Cake Details</h2>
              <div className="grid sm:grid-cols-2 gap-6">
                <div>
                  <p className="text-sm text-gray-600">Budget</p>
                  <p className="font-semibold text-gray-900">₦{order.budget.toLocaleString()}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Size</p>
                  <p className="font-semibold text-gray-900 capitalize">{order.cake_size}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Servings</p>
                  <p className="font-semibold text-gray-900">{order.servings} people</p>
                </div>
                {order.cake_message && (
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600">Cake Message</p>
                    <p className="font-semibold text-gray-900">{order.cake_message}</p>
                  </div>
                )}
                {order.special_requests && (
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600">Special Requests</p>
                    <p className="font-semibold text-gray-900">{order.special_requests}</p>
                  </div>
                )}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default TrackOrderPage;
