import { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useAuth } from '@/context/AuthContext';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { toast } from 'sonner';
import { ArrowLeft, Loader2, Car } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const CreateBooking = () => {
  const { providerId } = useParams();
  const navigate = useNavigate();
  const { user } = useAuth();
  const [provider, setProvider] = useState(null);
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    service_name: '',
    vehicle_type: 'sedan',
    vehicle_model: '',
    vehicle_plate: '',
    service_location: '',
    scheduled_date: '',
    special_instructions: '',
    total_price: 0
  });

  useEffect(() => {
    fetchProvider();
  }, [providerId]);

  const fetchProvider = async () => {
    try {
      const response = await axios.get(`${API}/providers/${providerId}`);
      setProvider(response.data);
    } catch (error) {
      toast.error('Failed to load provider');
    }
  };

  const calculatePrice = (serviceName, vehicleType) => {
    if (!provider || !serviceName || !vehicleType) return 0;
    
    const service = provider.services.find(s => s.name === serviceName);
    if (!service) return 0;
    
    if (vehicleType === 'sedan') return service.sedan_price;
    if (vehicleType === 'suv') return service.suv_price;
    if (vehicleType === 'truck') return service.truck_price;
    return 0;
  };

  useEffect(() => {
    const price = calculatePrice(formData.service_name, formData.vehicle_type);
    setFormData(prev => ({ ...prev, total_price: price }));
  }, [formData.service_name, formData.vehicle_type]);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);

    try {
      const bookingData = {
        ...formData,
        provider_id: providerId,
        total_price: formData.total_price,
        scheduled_date: new Date(formData.scheduled_date).toISOString()
      };

      await axios.post(`${API}/bookings`, bookingData);
      toast.success('Booking created successfully!');
      navigate('/my-bookings');
    } catch (error) {
      toast.error(error.response?.data?.detail || 'Failed to create booking');
    } finally {
      setLoading(false);
    }
  };

  if (!provider) {
    return <div className="flex items-center justify-center min-h-screen">Loading...</div>;
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <nav className="bg-white border-b border-gray-200 px-6 py-4">
        <div className="max-w-7xl mx-auto flex items-center">
          <Button variant="ghost" onClick={() => navigate(-1)}>
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back
          </Button>
        </div>
      </nav>

      <div className="max-w-4xl mx-auto px-6 py-8">
        <div className="bg-white rounded-xl shadow-sm p-8">
          <div className="flex items-center space-x-3 mb-2">
            <Car className="w-8 h-8 text-blue-600" />
            <h1 className="text-3xl font-bold">Book Car Wash Service</h1>
          </div>
          <p className="text-gray-600 mb-8">Booking with {provider.business_name} ({provider.service_type === 'mobile' ? 'Mobile Service' : 'Fixed Location'})</p>

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Service Selection */}
            <div>
              <Label htmlFor="service_name">Select Service *</Label>
              <Select
                value={formData.service_name}
                onValueChange={(value) => setFormData({...formData, service_name: value})}
                required
              >
                <SelectTrigger data-testid="service-select" className="mt-2">
                  <SelectValue placeholder="Choose a service" />
                </SelectTrigger>
                <SelectContent>
                  {provider.services.map((service, idx) => (
                    <SelectItem key={idx} value={service.name}>
                      {service.name} ({service.duration_minutes} min)
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Vehicle Type */}
            <div>
              <Label className="mb-3 block">Vehicle Type *</Label>
              <RadioGroup
                value={formData.vehicle_type}
                onValueChange={(value) => setFormData({...formData, vehicle_type: value})}
                className="grid grid-cols-3 gap-4"
              >
                <div className="flex items-center space-x-2 border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-blue-500">
                  <RadioGroupItem data-testid="vehicle-sedan" value="sedan" id="sedan" />
                  <Label htmlFor="sedan" className="flex-1 cursor-pointer">
                    <div className="font-semibold">Sedan</div>
                    {formData.service_name && (
                      <div className="text-sm text-blue-600 font-bold">₦{calculatePrice(formData.service_name, 'sedan')}</div>
                    )}
                  </Label>
                </div>
                
                <div className="flex items-center space-x-2 border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-blue-500">
                  <RadioGroupItem data-testid="vehicle-suv" value="suv" id="suv" />
                  <Label htmlFor="suv" className="flex-1 cursor-pointer">
                    <div className="font-semibold">SUV</div>
                    {formData.service_name && (
                      <div className="text-sm text-blue-600 font-bold">₦{calculatePrice(formData.service_name, 'suv')}</div>
                    )}
                  </Label>
                </div>
                
                <div className="flex items-center space-x-2 border-2 border-gray-200 rounded-lg p-4 cursor-pointer hover:border-blue-500">
                  <RadioGroupItem data-testid="vehicle-truck" value="truck" id="truck" />
                  <Label htmlFor="truck" className="flex-1 cursor-pointer">
                    <div className="font-semibold">Truck</div>
                    {formData.service_name && (
                      <div className="text-sm text-blue-600 font-bold">₦{calculatePrice(formData.service_name, 'truck')}</div>
                    )}
                  </Label>
                </div>
              </RadioGroup>
            </div>

            {/* Vehicle Details */}
            <div className="grid sm:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="vehicle_model">Vehicle Model *</Label>
                <Input
                  data-testid="vehicle-model-input"
                  id="vehicle_model"
                  placeholder="e.g. Toyota Camry 2020"
                  value={formData.vehicle_model}
                  onChange={(e) => setFormData({...formData, vehicle_model: e.target.value})}
                  required
                  className="mt-2"
                />
              </div>

              <div>
                <Label htmlFor="vehicle_plate">License Plate *</Label>
                <Input
                  data-testid="vehicle-plate-input"
                  id="vehicle_plate"
                  placeholder="e.g. ABC-123-XY"
                  value={formData.vehicle_plate}
                  onChange={(e) => setFormData({...formData, vehicle_plate: e.target.value})}
                  required
                  className="mt-2"
                />
              </div>
            </div>

            {/* Service Location */}
            <div>
              <Label htmlFor="service_location">
                {provider.service_type === 'mobile' ? 'Your Address *' : 'Our Location'}
              </Label>
              {provider.service_type === 'mobile' ? (
                <Input
                  data-testid="service-location-input"
                  id="service_location"
                  placeholder="Where should we come?"
                  value={formData.service_location}
                  onChange={(e) => setFormData({...formData, service_location: e.target.value})}
                  required
                  className="mt-2"
                />
              ) : (
                <Input
                  data-testid="service-location-input"
                  id="service_location"
                  value={provider.location.address}
                  readOnly
                  className="mt-2 bg-gray-50"
                />
              )}
            </div>

            {/* Scheduled Date */}
            <div>
              <Label htmlFor="scheduled_date">Preferred Date & Time *</Label>
              <Input
                data-testid="scheduled-date-input"
                id="scheduled_date"
                type="datetime-local"
                value={formData.scheduled_date}
                onChange={(e) => {
                  setFormData({...formData, scheduled_date: e.target.value});
                  if (provider.service_type === 'fixed') {
                    setFormData(prev => ({...prev, service_location: provider.location.address}));
                  }
                }}
                required
                className="mt-2"
              />
            </div>

            {/* Special Instructions */}
            <div>
              <Label htmlFor="special_instructions">Special Instructions (Optional)</Label>
              <Textarea
                data-testid="instructions-textarea"
                id="special_instructions"
                placeholder="Any specific areas to focus on? Scratches, stains, etc."
                value={formData.special_instructions}
                onChange={(e) => setFormData({...formData, special_instructions: e.target.value})}
                rows={3}
                className="mt-2"
              />
            </div>

            {/* Price Summary */}
            <div className="bg-blue-50 rounded-lg p-6">
              <div className="flex justify-between items-center mb-2">
                <span className="font-semibold">Service:</span>
                <span className="text-gray-700">{formData.service_name || 'Not selected'}</span>
              </div>
              <div className="flex justify-between items-center mb-2">
                <span className="font-semibold">Vehicle Type:</span>
                <span className="text-gray-700 capitalize">{formData.vehicle_type}</span>
              </div>
              <div className="border-t border-blue-200 my-3"></div>
              <div className="flex justify-between items-center">
                <span className="font-semibold text-lg">Total Price:</span>
                <span className="text-3xl font-bold text-blue-600">₦{formData.total_price.toFixed(2)}</span>
              </div>
            </div>

            {/* Submit */}
            <div className="flex justify-end space-x-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => navigate(-1)}
              >
                Cancel
              </Button>
              <Button
                data-testid="confirm-booking-btn"
                type="submit"
                disabled={loading || formData.total_price === 0}
                className="gradient-blue text-white px-8"
              >
                {loading ? <><Loader2 className="w-5 h-5 mr-2 animate-spin" />Creating...</> : 'Confirm Booking'}
              </Button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
};

export default CreateBooking;
